import { DirectiveLocation, GraphQLError, Kind, OperationTypeNode, specifiedDirectives, } from "graphql";
export function KnownDirectivesRule(context) {
    const locationsMap = new Map();
    const astDefinitions = context.getDocument().definitions;
    for (const def of astDefinitions) {
        if (def.kind === Kind.DIRECTIVE_DEFINITION) {
            locationsMap.set(def.name.value, new Set(def.locations.map((name) => name.value)));
        }
    }
    for (const specifiedDirective of specifiedDirectives) {
        if (!locationsMap.has(specifiedDirective.name)) {
            locationsMap.set(specifiedDirective.name, new Set(specifiedDirective.locations.map((loc) => String(loc))));
        }
    }
    return {
        Directive(node, _key, _parent, _path, ancestors) {
            const name = node.name.value;
            const locations = locationsMap.get(name);
            if (!locations) {
                context.reportError(new GraphQLError(`Unknown directive "@${name}".`, {
                    nodes: node,
                    extensions: {
                        code: "INVALID_GRAPHQL",
                    },
                }));
                return;
            }
            const candidateLocation = getDirectiveLocationForASTPath(ancestors);
            if (candidateLocation && !locations.has(candidateLocation)) {
                context.reportError(new GraphQLError(`Directive "@${name}" may not be used on ${candidateLocation}.`, {
                    nodes: node,
                    extensions: {
                        code: "INVALID_GRAPHQL",
                    },
                }));
            }
        },
    };
}
function getDirectiveLocationForASTPath(ancestors) {
    const appliedTo = ancestors[ancestors.length - 1];
    if (!("kind" in appliedTo)) {
        throw new Error("Expected a node");
    }
    switch (appliedTo.kind) {
        case Kind.OPERATION_DEFINITION:
            return getDirectiveLocationForOperation(appliedTo.operation);
        case Kind.FIELD:
            return DirectiveLocation.FIELD;
        case Kind.FRAGMENT_SPREAD:
            return DirectiveLocation.FRAGMENT_SPREAD;
        case Kind.INLINE_FRAGMENT:
            return DirectiveLocation.INLINE_FRAGMENT;
        case Kind.FRAGMENT_DEFINITION:
            return DirectiveLocation.FRAGMENT_DEFINITION;
        case Kind.VARIABLE_DEFINITION:
            return DirectiveLocation.VARIABLE_DEFINITION;
        case Kind.SCHEMA_DEFINITION:
        case Kind.SCHEMA_EXTENSION:
            return DirectiveLocation.SCHEMA;
        case Kind.SCALAR_TYPE_DEFINITION:
        case Kind.SCALAR_TYPE_EXTENSION:
            return DirectiveLocation.SCALAR;
        case Kind.OBJECT_TYPE_DEFINITION:
        case Kind.OBJECT_TYPE_EXTENSION:
            return DirectiveLocation.OBJECT;
        case Kind.FIELD_DEFINITION:
            return DirectiveLocation.FIELD_DEFINITION;
        case Kind.INTERFACE_TYPE_DEFINITION:
        case Kind.INTERFACE_TYPE_EXTENSION:
            return DirectiveLocation.INTERFACE;
        case Kind.UNION_TYPE_DEFINITION:
        case Kind.UNION_TYPE_EXTENSION:
            return DirectiveLocation.UNION;
        case Kind.ENUM_TYPE_DEFINITION:
        case Kind.ENUM_TYPE_EXTENSION:
            return DirectiveLocation.ENUM;
        case Kind.ENUM_VALUE_DEFINITION:
            return DirectiveLocation.ENUM_VALUE;
        case Kind.INPUT_OBJECT_TYPE_DEFINITION:
        case Kind.INPUT_OBJECT_TYPE_EXTENSION:
            return DirectiveLocation.INPUT_OBJECT;
        case Kind.INPUT_VALUE_DEFINITION: {
            const parentNode = ancestors[ancestors.length - 3];
            if (!("kind" in parentNode)) {
                throw new Error("Expected a node");
            }
            return parentNode.kind === Kind.INPUT_OBJECT_TYPE_DEFINITION
                ? DirectiveLocation.INPUT_FIELD_DEFINITION
                : DirectiveLocation.ARGUMENT_DEFINITION;
        }
    }
}
function getDirectiveLocationForOperation(operation) {
    switch (operation) {
        case OperationTypeNode.QUERY:
            return DirectiveLocation.QUERY;
        case OperationTypeNode.MUTATION:
            return DirectiveLocation.MUTATION;
        case OperationTypeNode.SUBSCRIPTION:
            return DirectiveLocation.SUBSCRIPTION;
    }
}
