import { createAdminApiClient } from '@shopify/admin-api-client';
import { logger } from '../../../logger/index.mjs';
import { MissingRequiredArgument } from '../../../error.mjs';
import { abstractFetch } from '../../../../runtime/http/index.mjs';
import '../../../../runtime/crypto/types.mjs';
import '../../../../runtime/crypto/crypto.mjs';
import { getUserAgent, clientLoggerFactory, throwFailedRequest } from '../../common.mjs';
import { canonicalizeHeaders } from '../../../../runtime/http/headers.mjs';

class GraphqlClient {
    static config;
    session;
    client;
    apiVersion;
    constructor(params) {
        const config = this.graphqlClass().config;
        if (!config.isCustomStoreApp && !params.session.accessToken) {
            throw new MissingRequiredArgument('Missing access token when creating GraphQL client');
        }
        if (params.apiVersion) {
            const message = params.apiVersion === config.apiVersion
                ? `Admin client has a redundant API version override to the default ${params.apiVersion}`
                : `Admin client overriding default API version ${config.apiVersion} with ${params.apiVersion}`;
            logger(config).debug(message);
        }
        this.session = params.session;
        this.apiVersion = params.apiVersion;
        this.client = createAdminApiClient({
            accessToken: config.adminApiAccessToken ?? this.session.accessToken,
            apiVersion: this.apiVersion ?? config.apiVersion,
            storeDomain: this.session.shop,
            customFetchApi: abstractFetch,
            logger: clientLoggerFactory(config),
            userAgentPrefix: getUserAgent(config),
            isTesting: config.isTesting,
        });
    }
    async query(params) {
        logger(this.graphqlClass().config).deprecated('12.0.0', 'The query method is deprecated, and was replaced with the request method.\n' +
            'See the migration guide: https://github.com/Shopify/shopify-app-js/blob/main/packages/apps/shopify-api/docs/migrating-to-v9.md#using-the-new-clients.');
        if ((typeof params.data === 'string' && params.data.length === 0) ||
            Object.entries(params.data).length === 0) {
            throw new MissingRequiredArgument('Query missing.');
        }
        let operation;
        let variables;
        if (typeof params.data === 'string') {
            operation = params.data;
        }
        else {
            operation = params.data.query;
            variables = params.data.variables;
        }
        const headers = Object.fromEntries(Object.entries(params?.extraHeaders ?? {}).map(([key, value]) => [
            key,
            Array.isArray(value) ? value.join(', ') : value.toString(),
        ]));
        const response = await this.request(operation, {
            headers,
            retries: params.tries ? params.tries - 1 : undefined,
            variables,
        });
        return { body: response, headers: {} };
    }
    async request(operation, options) {
        const response = await this.client.request(operation, {
            apiVersion: this.apiVersion || this.graphqlClass().config.apiVersion,
            ...options,
        });
        if (response.errors) {
            const fetchResponse = response.errors.response;
            throwFailedRequest(response, (options?.retries ?? 0) > 0, fetchResponse);
        }
        const headerObject = Object.fromEntries(response.headers ? response.headers.entries() : []);
        return {
            ...response,
            headers: canonicalizeHeaders(headerObject ?? {}),
        };
    }
    graphqlClass() {
        return this.constructor;
    }
}
function graphqlClientClass({ config, }) {
    class NewGraphqlClient extends GraphqlClient {
        static config = config;
    }
    Reflect.defineProperty(NewGraphqlClient, 'name', {
        value: 'GraphqlClient',
    });
    return NewGraphqlClient;
}

export { GraphqlClient, graphqlClientClass };
//# sourceMappingURL=client.mjs.map
