import { jsxs, Fragment, jsx } from 'react/jsx-runtime';
import { AppProvider as AppProvider$1 } from '@shopify/polaris';
import englishI18n from '@shopify/polaris/locales/en.json' with { type: 'json' };
import { APP_BRIDGE_URL } from '../../const.mjs';
import { RemixPolarisLink } from '../RemixPolarisLink.mjs';

/**
 * Sets up the Polaris `AppProvider` and injects the App Bridge script.
 *
 * This component extends the [`AppProvider`](https://polaris.shopify.com/components/utilities/app-provider) component
 * from Polaris, and accepts all of its props except for `linkComponent`, which is overridden to use the Remix `Link`
 * component.
 *
 * {@link https://polaris.shopify.com/components/utilities/app-provider}
 * {@link https://shopify.dev/tools/app-bridge}
 *
 * @example
 * <caption>Set up AppProvider.</caption>
 * <description>Wrap your app in the `AppProvider` component and pass in your API key.</description>
 * ```ts
 * // /app/routes/**\/*.ts
 * import {authenticate} from '~/shopify.server';
 * import {AppProvider} from '@shopify/shopify-app-remix/react';
 *
 * export async function loader({ request }) {
 *   await authenticate.admin(request);
 *
 *   return json({ apiKey: process.env.SHOPIFY_API_KEY });
 * }
 *
 * export default function App() {
 *   const { apiKey } = useLoaderData();
 *
 *   return (
 *     <AppProvider isEmbeddedApp apiKey={apiKey}>
 *       <Outlet />
 *     </AppProvider>
 *   );
 * }
 * ```
 *
 * @example
 * <caption>Localize Polaris components.</caption>
 * <description>Pass in a different locale for Polaris to translate its components.</description>
 * ```ts
 * // /app/routes/**\/*.ts
 * import {authenticate} from '~/shopify.server';
 * import {AppProvider} from '@shopify/shopify-app-remix/react';
 *
 * export async function loader({ request }) {
 *   await authenticate.admin(request);
 *
 *   return json({
 *     apiKey: process.env.SHOPIFY_API_KEY,
 *     polarisTranslations: require("@shopify/polaris/locales/fr.json"),
 *   });
 * }
 *
 * export default function App() {
 *   const { apiKey, polarisTranslations } = useLoaderData();
 *
 *   return (
 *     <AppProvider apiKey={apiKey} i18n={polarisTranslations}>
 *       <Outlet />
 *     </AppProvider>
 *   );
 * }
 * ```
 */
function AppProvider(props) {
    const { children, apiKey, i18n, isEmbeddedApp = true, __APP_BRIDGE_URL = APP_BRIDGE_URL, ...polarisProps } = props;
    return (jsxs(Fragment, { children: [isEmbeddedApp && jsx("script", { src: __APP_BRIDGE_URL, "data-api-key": apiKey }), jsx(AppProvider$1, { ...polarisProps, linkComponent: RemixPolarisLink, i18n: i18n || englishI18n, children: children })] }));
}

export { AppProvider };
//# sourceMappingURL=AppProvider.mjs.map
